<?php
/**
 * BwPostman Newsletter Component
 *
 * BwPostman single newsletter model for backend.
 *
 * @version 1.0.8.2 bwpm
 * @package BwPostman-Admin
 * @author Romana Boldt
 * @copyright (C) 2012-2015 Boldt Webservice <forum@boldt-webservice.de>
 * @support http://www.boldt-webservice.de/forum/bwpostman.html
 * @license GNU/GPL, see LICENSE.txt
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

// Check to ensure this file is included in Joomla!
defined ('_JEXEC') or die ('Restricted access');

require_once (JPATH_SITE.'/components/com_content/helpers/route.php');

// Import MODEL object class
jimport('joomla.application.component.modeladmin');

// Require helper class
require_once (JPATH_COMPONENT_ADMINISTRATOR.'/helpers/helper.php');

/**
 * BwPostman newsletter model
 * Provides methodes to add, edit and send newsletters
 *
 * @package		BwPostman-Admin
 * @subpackage	Newsletters
 */
class BwPostmanModelNewsletter extends JModelAdmin
{
	/**
	 * Newsletter id
	 *
	 * @var int
	 */
	var $_id = null;

	/**
	 * Newsletter data
	 *
	 * @var array
	 */
	var $_data = null;

	/**
	 * Constructor
	 * Determines the newsletter ID
	 *
	 */
	public function __construct()
	{
		parent::__construct();

		$jinput	= JFactory::getApplication()->input;
		$array	= $jinput->get('cid',  0, '', 'array');
		$this->setId((int)$array[0]);
	}

	/**
	 * Returns a Table object, always creating it.
	 *
	 * @param	type	The table type to instantiate
	 * @param	string	A prefix for the table class name. Optional.
	 * @param	array	Configuration array for model. Optional.
	 *
	 * @return	JTable	A database object
	 * 
	 * @since  1.0.1
	*/
	public function getTable($type = 'Newsletters', $prefix = 'BwPostmanTable', $config = array())
	{
		return JTable::getInstance($type, $prefix, $config);
	}

	/**
	 * Method to reset the newsletter ID and data
	 *
	 * @access	public
	 * @param	int Newsletter ID
	 */
	public function setId($id)
	{
		$this->_id	    = $id;
		$this->_data	= null;
	}

	/**
	 * Method to test whether a record can be deleted.
	 *
	 * @param	object	$record	A record object.
	 *
	 * @return	boolean	True if allowed to delete the record. Defaults to the permission set in the component.
	 * @since	1.0.1
	 */
	protected function canDelete($record)
	{
		$user = JFactory::getUser();

		// Check general delete permission first.
		if ($user->authorise('core.delete', 'com_bwpostman'))
		{
			return true;
		}
		
		if (!empty($record->id)) {
			// Check specific delete permission.
			if ($user->authorise('core.delete', 'com_bwpostman.newsletters.' . (int) $recordId))
			{
				return true;
			}
		}
		return false;
	}

	/**
	 * Method to test whether a record can have its state edited.
	 *
	 * @param	object	$record	A record object.
	 *
	 * @return	boolean	True if allowed to change the state of the record. Defaults to the permission set in the component.
	 * @since	1.0.1
	 */
	protected function canEditState($record)
	{
		$user = JFactory::getUser();

		// Check general edit state permission first.
		if ($user->authorise('core.edit.state', 'com_bwpostman'))
		{
			return true;
		}
		
		if (!empty($record->id)) {
			// Check specific edit state permission.
			if ($user->authorise('core.edit.state', 'com_bwpostman.newsletters.' . (int) $recordId))
			{
				return true;
			}
		}
		return false;
	}

	/**
	 * Method to get a single record.
	 *
	 * @param   integer  $pk  The id of the primary key.
	 *
	 * @return  mixed    Object on success, false on failure.
	 *
	 * @since   1.0.1
	 */
	public function getItem($pk = null)
	{
		$app		= JFactory::getApplication();
		$_db		= $this->_db;
		$query		= $_db->getQuery(true);
		$usergroups	= array();
		
		// Initialise variables.
		$pk		= (!empty($pk)) ? $pk : (int) $this->getState($this->getName() . '.id');
		$table	= $this->getTable();

			// Attempt to load the row.
			$return = $table->load($pk);

			// Check for a table object error.
			if ($return === false && $table->getError())
			{
				$this->setError($table->getError());
				return false;
			}
	
		// Convert to the JObject before adding other data.
		$properties	= $table->getProperties(1);
		$item 		= JArrayHelper::toObject($properties, 'JObject');
	
		if (property_exists($item, 'params')) {
			$registry = new JRegistry;
			$registry->loadJSON($item->params);
			$item->params = $registry->toArray();
		}
			
		// Convert the mailing date to local user time for display in the form.
		jimport('joomla.utilities.date');
		$tz = new DateTimeZone(JFactory::getApplication()->getCfg('offset'));

		if (intval($item->mailing_date)) {
			$date = new JDate($item->mailing_date);
			$date->setTimezone($tz);
			$item->mailing_date = $date->toSql(true);
		}
		else {
			$item->mailing_date = null;
		}
			
		//get associated mailinglists
       	$query->select($_db->quoteName('mailinglist_id'));
		$query->from($_db->quoteName('#__bwpostman_newsletters_mailinglists'));
		$query->where($_db->quoteName('newsletter_id') . ' = ' . (int) $item->id);
		$_db->setQuery($query);
		$item->mailinglists= $_db->loadColumn();

		//extract associated usergroups
		foreach ($item->mailinglists as $mailinglist) {
			if ((int) $mailinglist < 0) $usergroups[]	= -(int)$mailinglist;
		}
		$item->usergroups	= $usergroups;
		if ($pk == 0) $item->id	= 0;

		JFactory::getApplication()->setUserState('com_bwpostman.edit.newsletter.data', $item);

		return $item;
	}

	/**
	 * Method to get the record form.
	 *
	 * @param	array	$data		Data for the form.
	 * @param	boolean	$loadData	True if the form is to load its own data (default case), false if not.
	 *
	 * @return	mixed	A JForm object on success, false on failure
	 * @since	1.0.1
	 */
	public function getForm($data = array(), $loadData = true)
	{
		JForm::addFieldPath('JPATH_ADMINISTRATOR/components/com_bwpostman/models/fields');

		$params = JComponentHelper::getParams('com_bwpostman');
		$config = Jfactory::getConfig();
		$user	= JFactory::getUser();
		
		$empty_arr	= array();

		$form = $this->loadForm('com_bwpostman.newsletter', 'newsletter', array('control' => 'jform', 'load_data' => $loadData));
		if (empty($form)) {
			return false;
		}
		$jinput = JFactory::getApplication()->input;
		
		// The front end calls this model (@todo) and uses a_id to avoid id clashes so we need to check for that first.
		if ($jinput->get('a_id'))
		{
			$id =  $jinput->get('a_id', 0);
		}
		// The back end uses id so we use that the rest of the time and set it to 0 by default.
		else
		{
			$id =  $jinput->get('id', 0);
		}

		// predefine some values
		if (!$form->getValue('from_name')) {
			if ($params->get('default_from_name') && !$form->getValue('from_name')) {
				$form->setValue('from_name', '', $params->get('default_from_name'));
			}
			else {
				$form->setValue('from_name', '', $config->get('config.fromname'));
			}
		}
		
		if (!$form->getValue('from_email')) {
			if ($params->get('default_from_email')) {
				$form->setValue('from_email', '', $params->get('default_from_email'));
			}
			else {
				$form->setValue('from_email', '', $config->get('config.mailfrom'));
			}
		}
		
		if (!$form->getValue('reply_email')) {
			if ($params->get('default_reply_email')) {
				$form->setValue('reply_email', '', $params->get('default_reply_email'));
			}
			else {
				$form->setValue('reply_email', '', $config->get('config.mailfrom'));
			}
		}
		
		// Check for existing newsletter.
		// Modify the form based on Edit State access controls.
		if ($id != 0 && (!$user->authorise('core.edit.state', 'com_bwpostman.newsletter.'.(int) $id))
		|| ($id == 0 && !$user->authorise('core.edit.state', 'com_bwpostman'))
		)
		{
			// Disable fields for display.
			$form->setFieldAttribute('published', 'disabled', 'true');

			// Disable fields while saving.
			// The controller has already verified this is an newsletter you can edit.
			$form->setFieldAttribute('state', 'filter', 'unset');

		}
		// Check to show created data
		$c_date	= $form->getValue('created_date');
		if ($c_date == '0000-00-00 00:00:00') {
			$form->setFieldAttribute('created_date', 'type', 'hidden');
			$form->setFieldAttribute('created_by', 'type', 'hidden');
		}
		
		// Check to show modified data
		$m_date	= $form->getValue('modified_time');
		if ($m_date == '0000-00-00 00:00:00') {
			$form->setFieldAttribute('modified_time', 'type', 'hidden');
			$form->setFieldAttribute('modified_by', 'type', 'hidden');
		}
		
		// Check to show mailing data
		$s_date	= $form->getValue('mailing_date');
		if ($s_date == '0000-00-00 00:00:00') {
			$form->setFieldAttribute('mailing_date', 'type', 'hidden');
		}
		return $form;
	}

	/**
	/**
	 * Method to get the data that should be injected in the form.
	 *
	 * @return	mixed	The data for the form.
	 * @since	1.0.1
	 */
	protected function loadFormData()
	{
		// @todo XML-file will not be processed

		// Check the session for previously entered form data.
		$data		= JFactory::getApplication()->getUserState('com_bwpostman.edit.newsletter.data', null);
		
		if (empty($data)) {
			$data = $this->getItem();
		}
		else {
			if (property_exists($data, 'selected_content') !== true)	$data->selected_content = '';
			if (property_exists($data, 'usergroups') !== true)			$data->usergroups = array();
			
			if ((!$data->selected_content || $data->selected_content == '') && ($data->html_version == '') && ($data->text_version == '')) {
				$data->html_version	= '';
				$data->text_version	= '';
			}
	
			// merge ml-arrays, single array may not exist, therefore array_merge would not give a result
			if (property_exists($data, 'ml_available') && is_array($data->ml_available) && !empty($data->ml_available))			foreach ($data->ml_available as $key => $value) 	$data->mailinglists[] 	= $value;
			if (property_exists($data, 'ml_unavailable') && is_array($data->ml_unavailable) && !empty($data->ml_unavailable))	foreach ($data->ml_unavailable as $key => $value)	$data->mailinglists[] 	= $value;
			if (property_exists($data, 'ml_intern') && is_array($data->ml_intern) && !empty($data->ml_intern))					foreach ($data->ml_intern as $key => $value)		$data->mailinglists[] 	= $value;
		}
		return $data;
	}
		
	/**
	 * Method to get the data of a single newsletter for the preview/modalbox
	 *
	 * @access	public
	 * @return 	object Newsletter with formatted pieces
	 */
	public function getSingleNewsletter ()
	{
		$app			= JFactory::getApplication();
		$item			= $app->getUserState('com_bwpostman.edit.newsletter.data');
		
		//convert to object if necessary
		if ($item && !is_object($item)) {
			$data_tmp	= new stdClass();
			foreach ($item as $key => $value) {
				$data_tmp->$key	= $value;
			}
			$item = $data_tmp;
		}
		
		if ($item->id == 0 && !empty($item->selected_content) && empty($item->html_version) && empty($item->text_version)) {
			if (!is_array($item->selected_content)) $item->selected_content = explode(',', $item->selected_content);
			$renderer	= new contentRenderer();
			$content	= $renderer->getContent((array) $item->selected_content, $item->subject);
			$item->html_version	= $content['html_version'];
			$item->text_version	= $content['text_version'];
		}

		// Replace the links to provide the correct preview
		$item->html_formatted	= $item->html_version;
		$item->text_formatted	= $item->text_version;

		$this->_replaceLinks($item->html_formatted);
		$this->_replaceLinks($item->text_formatted);

		$this->_addHtmlTags($item->html_formatted);
		$this->_addHTMLFooter($item->html_formatted);
		$this->_addTextFooter($item->text_formatted);
		
		return $item;
	}

	/**
	 * Method to get the selected content items which are used to compose a newsletter
	 *
	 * @access	public
	 * @return	array
	 */
	public function getSelectedContent()
	{
		$_db	= $this->_db;
		
		$selected_content = $this->_selectedContent();
		$selected_content_void = array ();

		if ($selected_content) {
			if (!is_array($selected_content)) $selected_content = explode(',',$selected_content);
							
			$selected_content_items = array();
				
			// We do a foreach to protect our ordering
			foreach($selected_content as $content_id){
				$subquery	= $_db->getQuery(true);
				$subquery->select($_db->quoteName('cc') . '.' . $_db->quoteName('title'));
				$subquery->from($_db->quoteName('#__categories') . ' AS ' . $_db->quoteName('cc'));
				$subquery->where($_db->quoteName('cc') . '.' . $_db->quoteName('id') . ' = ' . $_db->quoteName('c') . '.' . $_db->quoteName('catid'));
				
				$query	= $_db->getQuery(true);
				$query->select($_db->quoteName('c') . '.' . $_db->quoteName('id'));
				$query->select($_db->quoteName('c') . '.' . $_db->quoteName('title') . ', (' . $subquery) . ') AS ' . $_db->quoteName('category_name');
				$query->from($_db->quoteName('#__content') . ' AS ' . $_db->quoteName('c'));
				$query->where($_db->quoteName('c') . '.' . $_db->quoteName('id') . ' = ' . $_db->Quote($content_id));
				
				$_db->setQuery($query);
				
				$items= $_db->loadObjectList();
				 
				if(sizeof($items) > 0){
					if ($items[0]->category_name == '') {
						$selected_content_items[] = JHTML::_('select.option', $items[0]->id, "Uncategorized - " . $items[0]->title);
					} else {
						$selected_content_items[] = JHTML::_('select.option', $items[0]->id, $items[0]->category_name . " - " . $items[0]->title);
					}
				}
			}
			return $selected_content_items;

		} 
		else {
			return $selected_content_void;
		}
	}

	/**
	 * Method to get the menu item ID which will be needed for the unsubscribe link in the footer
	 *
	 * @access	public
	 * @return 	int menu item ID
	 */
	public function getItemid($view)
	{
		$_db	= JFactory::getDbo();
		$query	= $_db->getQuery(true);
		
		$query->select($_db->quoteName('id'));
		$query->from($_db->quoteName('#__menu'));
		$query->where($_db->quoteName('link') . ' = ' . $_db->Quote('index.php?option=com_bwpostman&view='.$view));
		$query->where($_db->quoteName('published') . ' = ' . (int) 1);
				
		$_db->setQuery($query);
		$itemid = $_db->loadResult();

		if (empty($itemid)) {
			$query	= $_db->getQuery(true);
			
			$query->select($_db->quoteName('id'));
			$query->from($_db->quoteName('#__menu'));
			$query->where($_db->quoteName('link') . ' = ' . $_db->Quote('index.php?option=com_bwpostman&view=register'));
			$query->where($_db->quoteName('published') . ' = ' . (int) 1);
					
			$_db->setQuery($query);
			$itemid = $_db->loadResult();
		}
		return $itemid;
	}

	/**
	 * Method to get the language of an article
	 *
	 * @access	public
	 * 
	 * @param	int		article ID
	 * 
	 * @return 	mixed	language string or 0
	 * 
	 * @since	1.0.7
	 */
	static public function getArticleLanguage($id)
	{
		if (JLanguageMultilang::isEnabled()) {
			$_db	= JFactory::getDbo();
			$query	= $_db->getQuery(true);
			
			$query->select($_db->quoteName('language'));
			$query->from($_db->quoteName('#__content'));
			$query->where($_db->quoteName('id') . ' = ' . (int) $id);
					
			$_db->setQuery($query);
			$result = $_db->loadResult();
	
			return $result;
		}
		else {
			return 0;
		}
	}

	/**
	 * Method to store the newsletter data from the newsletters_tmp-table into the newsletters-table
	 *
	 * @access	public
	 * @return 	boolean
	 */
	public function save($data)
	{
		$app	= JFactory::getApplication();
		$_db	= $this->_db;
		$query	= $_db->getQuery(true);
		$fault	= false;
		
		// merge ml-arrays, single array may not exist, therefore array_merge would not give a result
		if (isset($data['ml_available']) && !empty($data['ml_available']))		foreach ($data['ml_available'] as $key => $value) 	$data['mailinglists'][] 	= $value;
		if (isset($data['ml_unavailable']) && !empty($data['ml_unavailable']))	foreach ($data['ml_unavailable'] as $key => $value)	$data['mailinglists'][] 	= $value;
		if (isset($data['ml_intern']) && !empty($data['ml_intern']))			foreach ($data['ml_intern'] as $key => $value)		$data['mailinglists'][] 	= $value;

		// merge usergroups into mailinglists, single array may not exist, therefore array_merge would not give a result
		if (isset($data['usergroups']) && !empty($data['usergroups']))		foreach ($data['usergroups'] as $key => $value)		$data['mailinglists'][] 	= '-' . $value;
		
		if ((!isset($data['mailinglists']) || empty ($data['mailinglists'])) && (!isset($data['usergroups']) || empty ($data['usergroups']))) {
			$app->enqueueMessage(JText::_('COM_BWPOSTMAN_NL_ERROR_SAVE_NO_RECIPIENT'), 'error');
			// construct with $fault to remind done entries in state
			$fault = true;
		}

		// if saving a new newsletter before changing tab, we have to look, if there is a content selected and set html- and text-version 
		if (empty($data['html_version']) && empty($data['text_version'])) {
			if (isset($data['selected_content'])) {
				// only render new content, if selection from article list is present
				$renderer	= new contentRenderer();
				$content	= $renderer->getContent($data['selected_content'], $data['subject']);
				
				$data['html_version']	= $content['html_version'];
				$data['text_version']	= $content['text_version'];
			}
			else { //  no selected content
				$app->enqueueMessage(JText::_('COM_BWPOSTMAN_NL_ERROR_CONTENT_MISSING'), 'error');
				// construct with $fault to remind done entries in state
				$fault = true;
			}
		}
			
		if (isset($data['selected_content'])) {
			if (is_array($data['selected_content'])) $data['selected_content']	= implode(',', $data['selected_content']);
		}
		
		// convert data to object for storing in state before saving
		if (is_array($data)) {
			$state_data = new stdClass();
			foreach ($data as $k => $v) {
					$state_data->$k = $v;
			}
		}
		
		// remind done entries in state
		$app->setUserState('com_bwpostman.edit.newsletter.data', $state_data);
		
		// construct with $fault to remind done entries in state  
		if ($fault) return false;
		
		if (!parent::save($data)) {
			return false;
		}
		
		// Delete all entrys of the newsletter from newsletters_mailinglists-table
		if ($data['id']){
			$query->delete($_db->quoteName('#__bwpostman_newsletters_mailinglists'));
			$query->where($_db->quoteName('newsletter_id') . ' =  ' . (int) $data['id']);
			
			$_db->setQuery($query);
			$_db->Execute($query);
		}
		else {
			//get id of new inserted data to write cross table subscribers-mailinglists and update state
			$data['id']= $this->getState('newsletter.id');
			$state_data->id	= $data['id'];
			$app->setUserState('com_bwpostman.edit.newsletter.data', $state_data);
		}
		
		// Store the selected BwPostman mailinglists into newsletters_mailinglists-table
		if (isset($data['mailinglists'])) {
			foreach ($data['mailinglists'] AS $mailinglists_value) {
				$query	= $_db->getQuery(true);
								
				$query->insert($_db->quoteName('#__bwpostman_newsletters_mailinglists'));
				$query->columns(array(
					$_db->quoteName('newsletter_id'),
					$_db->quoteName('mailinglist_id')
					));
					$query->values(
					(int) $data['id'] . ',' .
					(int) $mailinglists_value
					);
				$_db->setQuery($query);
				$_db->execute();
			}
		}
		return true;
	}

	/**
	 * Method to (un)archive a newsletter from the newsletters-table
	 * --> when unarchiving it is called by the archive-controller
	 *
	 * @access	public
	 * @param	array Newsletter IDs
	 * @param	tinyint Task --> 1 = archive, 0 = unarchive
	 * @return	boolean
	 */
	public function archive($cid = array(), $archive = 1)
	{
		$app		= JFactory::getApplication();
		$date		= JFactory::getDate();
		$uid		= JFactory::getUser()->get('id');
		$state_data	= $app->getUserState('com_bwpostman.edit.newsletter.data');
		$_db		= $this->_db;
		$query		= $_db->getQuery(true);
		
		if ($archive == 1) {
			$time = $date->toSql();

			// Access check.
			foreach ($cid as $i) {
				$data = self::getItem($i);
				if (!BwPostmanHelper::allowArchive($i, $data->created_by, 'newsletter'))
				{
					$app->enqueueMessage(JText::_('COM_BWPOSTMAN_NL_ARCHIVE_RIGHTS_MISSING'), 'error');
					return false;
				}
			}
		} else {
			$time	= '0000-00-00 00:00:00';
			$uid	= 0;
		
			// Access check.
			foreach ($cid as $i) {
				$data = self::getItem($i);
				if (!BwPostmanHelper::allowRestore($i, $data->created_by, 'newsletter'))
				{
					$app->enqueueMessage(JText::_('COM_BWPOSTMAN_NL_RESTORE_RIGHTS_MISSING'), 'error');
					return false;
				}
			}
		}

		if (count($cid))
		{
			JArrayHelper::toInteger($cid);

			$query->update($_db->quoteName('#__bwpostman_newsletters'));
			$query->set($_db->quoteName('archive_flag') . " = " . (int) $archive);
			$query->set($_db->quoteName('archive_date') . " = " . $_db->Quote($time, false));
			$query->set($_db->quoteName('archived_by') . " = " . (int) $uid);
			$query->where($_db->quoteName('id') . ' IN (' .implode(',', $cid) . ')');
			
			$_db->setQuery($query);

			if (!$_db->query()) {
				$this->setError($_db->getErrorMsg());
				return false;
			}
		}
		$app->setUserState('com_bwpostman.edit.newsletter.data', $state_data);
		return true;
	}

	/**
	 * Method to copy one or more newsletters
	 * --> the assigned mailingslists will be copied, too
	 *
	 * @param 	array Newsletter-IDs
	 * @return 	boolean
	 */
	public function copy($cid = array())
	{
		$app	= JFactory::getApplication();
		$_db	= $this->_db;
		
		if (count($cid)) {
			foreach ($cid as $id){
				$newsletters	= $this->getTable('newsletters', 'BwPostmanTable');
				$query			= $_db->getQuery(true);
				
				$query->select('*');
				$query->from($_db->quoteName('#__bwpostman_newsletters'));
				$query->where($_db->quoteName('id') . ' = ' . (int) $id);
				
				$_db->setQuery($query);
				
				$newsletters_data_copy = $_db->loadObject();
				if (is_string($newsletters_data_copy->usergroups)) {
					if ($newsletters_data_copy->usergroups == '') {
						$newsletters_data_copy->usergroups = array();
					}
					else {
						$newsletters_data_copy->usergroups	= explode(',', $newsletters_data_copy->usergroups);
					}
				}
				
				if (!is_object($newsletters_data_copy)) {
					$app->enqueueMessage(JText::_('COM_BWPOSTMAN_NL_COPY_FAILED'), 'error');
				}

				$date	= JFactory::getDate();
				$time	= $date->toSql();
				$user	= JFactory::getUser();
				$uid	= $user->get('id');

				$newsletters_data_copy->id 					= null;
				$newsletters_data_copy->asset_id			= null;
				$newsletters_data_copy->subject 			= JText::sprintf('COM_BWPOSTMAN_NL_COPY_OF', $newsletters_data_copy->subject); 
				$newsletters_data_copy->attachment	 		= null;
				$newsletters_data_copy->created_date 		= $time;
				$newsletters_data_copy->created_by			= $uid;
				$newsletters_data_copy->modified_time	 	= $time;
				$newsletters_data_copy->modified_by	 		= null;
				$newsletters_data_copy->mailing_date 		= 0;
				$newsletters_data_copy->published 			= null;
				$newsletters_data_copy->checked_out 		= 0;
				$newsletters_data_copy->checked_out_time 	= 0;
				$newsletters_data_copy->archive_flag 		= 0;
				$newsletters_data_copy->archive_date 		= 0;
				$newsletters_data_copy->hits 				= null;
				
				$subQuery	= $_db->getQuery(true);
				
				$subQuery->select($_db->quoteName('mailinglist_id'));
				$subQuery->from($_db->quoteName('#__bwpostman_newsletters_mailinglists'));
				$subQuery->where($_db->quoteName('newsletter_id') . ' = ' . (int) $id);

				$_db->setQuery($subQuery);
				
				$newsletters_data_copy->mailinglists	= $_db->loadColumn();

				if (!$this->save(JArrayHelper::fromObject($newsletters_data_copy, false))) {
					$app->enqueueMessage($_db->getErrorMsg(), 'error');
					return false;
				}
			}
			$app->enqueueMessage(JText::_('COM_BWPOSTMAN_NL_COPIED'), 'message');
			return true;
		}
		else {
			$app->enqueueMessage(JText::_('COM_BWPOSTMAN_NL_ERROR_COPYING'), 'error');
			return false;
		}
	}

	/**
	 * Method to remove one or more newsletters from the newsletters-table
	 * --> is called by the archive-controller
	 *
	 * @access	public
	 * @param	array Newsletter IDs
	 * @return	boolean
	 */
	public function delete(&$pks)
	{
		$result = false;

		// Access check.
		foreach ($pks as $i) {
			$data = self::getItem($i);
			if (!BwPostmanHelper::allowDelete($i, $data->created_by, 'newsletter'))
			{
				return false;
			}
		}

		if (count($pks))
		{
			JArrayHelper::toInteger($pks);
			$cids = implode(',', $pks);

			// Delete newsletter from newsletters-table
			$nl_table = JTable::getInstance('newsletters', 'BwPostmanTable');

			foreach ($pks as $id) {
				if (!$nl_table->delete($id))
				{
					return false;
				}
			}
				
			// Delete assigned mailinglists from newsletters_mailinglists-table
			$lists_table = JTable::getInstance('newsletters_mailinglists', 'BwPostmanTable');
			
			foreach ($pks as $id) {
				if (!$lists_table->delete($id))
				{
					return false;
				}
			}
		}
		return true;
	}

	/**
	 * Method to clear the queue
	 *
	 * @access	public
	 * @return 	boolean
	 */
	public function delete_queue()
	{
		$_db	= $this->_db;
		$query	= $_db->getQuery(true);
		
		$query = "TRUNCATE TABLE {$_db->quoteName('#__bwpostman_sendmailqueue')} ";
		$_db->setQuery($query);
		if(!$_db->query()) {
			$this->setError($_db->getErrorMsg());
			return false;
		} else {
			return true;
		}
	}

	/**
	 * Method to check and clean the input fields
	 *
	 * @access	public
	 * @return	boolean
	 */
	public function checkForm($data, &$err)
	{
		jimport('joomla.mail.helper');

		$i = 0;

		//Remove all HTML tags from name, emails, subject and the text version
		$filter = new JFilterInput(array(), array(), 0, 0);
		$data['from_name'] 		= $filter->clean($data['from_name']);
		$data['from_email'] 	= $filter->clean($data['from_email']);
		$data['reply_email'] 	= $filter->clean($data['reply_email']);
		$data['subject']		= $filter->clean($data['subject']);
		$data['text_version']	= $filter->clean($data['text_version']);

		$err = array();

		// Check for valid from_name
		if (trim($data['from_name']) == '') {
			$err[$i]['err_code'] = 301;
			$err[$i]['err_msg'] = JText::_('COM_BWPOSTMAN_NL_ERROR_FROM_NAME');
			$i++;
		}

		// Check for valid from_email address
		if (trim($data['from_email']) == '') {
			$err[$i]['err_code'] = 302;
			$err[$i]['err_msg'] = JText::_('COM_BWPOSTMAN_NL_ERROR_FROM_EMAIL');
			$i++;
		} else {
			// If there is a from_email adress check if the adress is valid
			if (!JMailHelper::isEmailAddress(trim($data['from_email']))) {
				$err[$i]['err_code'] = 306;
				$err[$i]['err_msg'] = JText::_('COM_BWPOSTMAN_NL_ERROR_FROM_EMAIL_INVALID');
				$i++;
			}
		}

		// Check for valid reply_email address
		if (trim($data['reply_email']) == '') {
			$err[$i]['err_code'] = 303;
			$err[$i]['err_msg'] = JText::_('COM_BWPOSTMAN_NL_ERROR_REPLY_EMAIL');
			$i++;
		} else {
			// If there is a from_email adress check if the adress is valid
			if (!JMailHelper::isEmailAddress(trim($data['reply_email']))) {
				$err[$i]['err_code'] = 307;
				$err[$i]['err_msg'] = JText::_('COM_BWPOSTMAN_NL_ERROR_REPLY_EMAIL_INVALID');
				$i++;
			}
		}

		// Check for valid subject
		if (trim($data['subject']) == '') {
			$err[$i]['err_code'] = 304;
			$err[$i]['err_msg'] = JText::_('COM_BWPOSTMAN_NL_ERROR_SUBJECT');
			$i++;
		}

		// Check for valid html or text version
		if ((trim($data['html_version']) == '') && (trim($data['text_version']) == ''))  {
			$err[$i]['err_code'] = 305;
			$err[$i]['err_msg'] = JText::_('COM_BWPOSTMAN_NL_ERROR_HTML_AND_TEXT');
		}
		return;
	}

	/**
	 * Method to check if there are selected mailinglists and if they contain recipients
	 *
	 * @access	public
	 * @param	string Error message
	 * @param	boolean	Status --> 0 = do not send to unconfirmed, 1 = sent also to unconfirmed
	 * @return 	object Test-recipients data
	 */
	public function checkRecipients(&$ret_msg, $nl_id, $send_to_unconfirmed)
	{
		$_db	= $this->_db;
		$query	= $_db->getQuery(true);
		
		// Check if there are assigned mailinglists or usergroups
		$query->select($_db->quoteName('mailinglist_id'));
		$query->from($_db->quoteName('#__bwpostman_newsletters_mailinglists'));
		$query->where($_db->quoteName('newsletter_id') . ' = ' . (int) $nl_id);
		
		$_db->setQuery($query);
		
		$mailinglists = $_db->loadObjectList();
		
		if (!$mailinglists) {
			$ret_msg = JText::_('COM_BWPOSTMAN_NL_ERROR_SENDING_NL_NO_LISTS');
			return false;
		}
			
		$check_subscribers		= 0;
		$check_allsubscribers	= 0;
		$count_users			= 0;
		$usergroup				= array();
		
		foreach ($mailinglists as $mailinglist){
			$mailinglist_id = $mailinglist->mailinglist_id;
			// Mailinglists
			if ($mailinglist_id > 0) $check_subscribers = 1;
			// All subscribers
			if ($mailinglist_id == -1) {
				$check_allsubscribers = 1;
			}
			else {
				// Usergroups
				if ((int) $mailinglist_id < 0) $usergroup[] = -(int) $mailinglist_id;
			}
		}

		// Check if the subscribers are confirmed and not archived
		if ($check_subscribers){ // Check subscribers from selected mailinglists
				
			if ($send_to_unconfirmed) {
				$status = '0,1';
			} else {
				$status = '1';
			}
			
			$subQuery1	= $_db->getQuery(true);
			$subQuery2	= $_db->getQuery(true);
			$query		= $_db->getQuery(true);
			
			$subQuery2->select($_db->quoteName('mailinglist_id'));
			$subQuery2->from($_db->quoteName('#__bwpostman_newsletters_mailinglists'));
			$subQuery2->where($_db->quoteName('newsletter_id') . ' IN (' . (int) $nl_id . ')');
			
			$subQuery1->select('DISTINCT' . $_db->quoteName('subscriber_id'));
			$subQuery1->from($_db->quoteName('#__bwpostman_subscribers_mailinglists'));
			$subQuery1->where($_db->quoteName('mailinglist_id') . ' IN (' . $subQuery2 . ')');
			
			$query->select('COUNT(' . $_db->quoteName('id') . ')');
			$query->from($_db->quoteName('#__bwpostman_subscribers'));
			$query->where($_db->quoteName('id') . ' IN (' . $subQuery1 . ')');
			$query->where($_db->quoteName('status') . ' IN (' . $status . ')');
			$query->where($_db->quoteName('archive_flag') . ' = ' . (int) 0);
			
			$_db->setQuery($query);
			
			$count_subscribers = $_db->loadResult();
		} 
		elseif ($check_allsubscribers){ // Check all subscribers (select option "All subscribers")
				
			if ($send_to_unconfirmed) {
				$status = '0,1,9';
			} else {
				$status = '1,9';
			}
				
			$query		= $_db->getQuery(true);
			
			$query->select('COUNT(' . $_db->quoteName('id') . ')');
			$query->from($_db->quoteName('#__bwpostman_subscribers'));
			$query->where($_db->quoteName('status') . ' IN (' . $status . ')');
			$query->where($_db->quoteName('archive_flag') . ' = ' . (int) 0);
			
			$_db->setQuery($query);
			
			$count_subscribers = $_db->loadResult();
		}

		// Checks if the selected usergroups contain users
		if (is_array($usergroup) && count($usergroup)){
			$count_users = 0;
			$query		= $_db->getQuery(true);
			$sub_query	= $_db->getQuery(true);
			
			$sub_query->select($_db->quoteName('g') . '.' . $_db->quoteName('user_id'));
			$sub_query->from($_db->quoteName('#__user_usergroup_map') . ' AS ' . $_db->quoteName('g'));
			$sub_query->where($_db->quoteName('g') . '.' . $_db->quoteName('group_id') . ' IN (' . implode(',', $usergroup) . ')');
			
			$query->select('COUNT(' . $_db->quoteName('u') . '.' . $_db->quoteName('id') . ')');
			$query->from($_db->quoteName('#__users') . ' AS ' . $_db->quoteName('u'));
			$query->where($_db->quoteName('u') . '.' . $_db->quoteName('block') . ' = ' . (int) 0);
			$query->where($_db->quoteName('u') . '.' . $_db->quoteName('activation') . ' = ' . $_db->Quote(''));
			$query->where($_db->quoteName('u') . '.' . $_db->quoteName('id') . ' IN (' . $sub_query . ')');
			
			$_db->setQuery($query);
			$count_users = $_db->loadResult();
		}

		// We return only false, if no subscribers AND no joomla users are selected.
		if (!$count_users && !$count_subscribers){
			if (!$count_users) {
				$ret_msg = JText::_('COM_BWPOSTMAN_NL_ERROR_SENDING_NL_NO_USERS');
				return false;
			}
			if (!$count_subscribers) {
				$ret_msg = JText::_('COM_BWPOSTMAN_NL_ERROR_SENDING_NL_NO_SUBSCRIBERS');
				return false;
			}
		}
		return true;
	}

	/**
	 * Method to check if there are test-recipients to whom the newsletter shall be send
	 *
	 * @access	public
	 * @return 	boolean
	 */
	public function checkTestrecipients()
	{
		$_db	= $this->_db;
		$query	= $_db->getQuery(true);
		
		$query->select('COUNT(' . $_db->quoteName('id') . ')');
		$query->from($_db->quoteName('#__bwpostman_subscribers'));
		$query->where($_db->quoteName('status') . ' = ' . (int) 9);
		$query->where($_db->quoteName('archive_flag') . ' = ' . (int) 0);
		
		$_db->setQuery($query);
		
		$testrecipients = $_db->loadResult();

		if ($testrecipients) {
			return true;
		} else {
			return false;
		}
	}

	/**
	 * Method to compose a newsletter out of the selected content items
	 *
	 * @access	public
	 * @return 	associcative array of Content data
	 */
	public function composeNl()
	{
		$jinput	= JFactory::getApplication()->input;
		
		$nl_content = $jinput->get('selected_content');
		$nl_subject = $jinput->get('subject');
		$renderer	= new contentRenderer();
		$content	= $renderer->getContent($nl_content, $nl_subject);

		return $content;
	}

	/**
	 * Method to fetch the content out of the selected content items
	 *
	 * @access	public
	 * @return 	associcative array of Content data
	 */
	public function changeTab()
	{
		$app			= JFactory::getApplication();
		$jinput			= JFactory::getApplication()->input;
		$form_data		= $jinput->get('jform', '', 'array');
		$content_exists	= $jinput->get('content_exists', '', 'string');
		$add_content	= $jinput->get('add_content', 0);
		$sel_content	= $jinput->get('selected_content_old', '', 'string');
		$state_data		= $app->getUserState('com_bwpostman.edit.newsletter.data');

		if (array_key_exists('selected_content', $form_data) !== true) $form_data['selected_content'] = array();
		if (array_key_exists('usergroups', $form_data) !== true) $form_data['usergroups'] = array();
				
		// merge ml-arrays, single array may not exist, therefore array_merge would not give a result
		if (isset($form_data['ml_available']))		foreach ($form_data['ml_available'] as $key => $value) 		$form_data['mailinglists'][] 	= $value;
		if (isset($form_data['ml_unavailable']))	foreach ($form_data['ml_unavailable'] as $key => $value)	$form_data['mailinglists'][] 	= $value;
		if (isset($form_data['ml_intern']))			foreach ($form_data['ml_intern'] as $key => $value)			$form_data['mailinglists'][] 	= $value;
		
		$nl_content = (array) $form_data['selected_content'];
		$nl_subject = $form_data['subject'];
		
		if (is_array($form_data['selected_content'])) $form_data['selected_content']	= implode(',', $form_data['selected_content']);

		if ($add_content) { // some content has changed
			if ($sel_content != $form_data['selected_content']) { 

				// no content is selected
				if ($add_content == -1  && empty($nl_content)) $nl_content =  (array) "-1";

				// only render new content, if selection from article list has changed
				$renderer	= new contentRenderer();
				$content	= $renderer->getContent($nl_content, $nl_subject);
				
				$form_data['html_version']			= $content['html_version'];
				$form_data['text_version']			= $content['text_version'];
				$form_data['selected_content_old']	= $form_data['selected_content'];
			}
		}
		else {
			$form_data['selected_content']	= $state_data->selected_content;
		}
		
		// convert form data to object to update state
		$data = new stdClass();
		foreach ($form_data as $k => $v)
		{
			$data->$k = $v;
		}

		$app->setUserState('com_bwpostman.edit.newsletter.data', $data);
		$app->setUserState('com_bwpostman.edit.newsletter.changeTab', true);

		return;
	}
		
	/**
	 * Method to prepare the sending of a newsletter
	 *
	 * @access	public
	 *
	 * @param 	string	Recipient --> either recipients or test-recipients
	 * @param 	int		Newsletter ID
	 * @param 	boolean	Send to unconfirmed or not
	 */
	public function sendNewsletter($recipients, $nl_id, $unconfirmed)
	{
		// Prepare the newsletter content
		$id	= $this->_addSendMailContent($nl_id, $recipients);

		// Prepare the recipient queue
		$this->_addSendMailQueue($id, $recipients, $nl_id, $unconfirmed);

		// Update the newsletters table, to prevent repeated sending of the newsletter
		if ($recipients == 'recipients') {
			$tblNewsletters = $this->getTable('newsletters', 'BwPostmanTable');
			$tblNewsletters->markAsSent($nl_id);
		}

		// The actual sending of the newsletter is executed only in
		// Sendmail Queue layout.
	}

	/**
	 * Method to reset the count of sending attempts in sendmailqueue.
	 *
	 * @return unknown_type
	 */
	public function resetSendAttempts()
	{
		$tblSendmailQueue = $this->getTable('sendmailqueue', 'BwPostmanTable');
		$tblSendmailQueue->resetTrials();
	}

	/**
	 * Method to get the selected content
	 *
	 * @access	public
	 * @return	string
	 */
	public function _selectedContent()
	{
		$_db	= $this->_db;
		
		// Get selected content from the newsletters-Table
		$query	= $_db->getQuery(true);
		
		$query->select($_db->quoteName('selected_content'));
		$query->from($_db->quoteName('#__bwpostman_newsletters'));
		$query->where($_db->quoteName('id') . ' = ' . (int) JFactory::getApplication()->getUserState('com_bwpostman.edit.newsletter.data.id'));
		
		$_db->setQuery($query);
		$content_ids = $_db->loadResult();

		return $content_ids;
	}
	
	/**
	 * Method to replace the links in a newsletter to provide the correct preview
	 *
	 * @access	private
	 * @param 	string HTML-/Text-version
	 * @return 	boolean
	 */
	private function _replaceLinks(&$text)
	{
		$search_str = '/\s+(href|src)\s*=\s*["\']?\s*(?!http|mailto)([\w\s&%=?#\/\.;:_-]+)\s*["\']?/i';
		$text = preg_replace($search_str,' ${1}="'.JURI::root().'${2}"',$text);
		return true;
	}
	
	/**
	 * Method to add the HTML-Tags and the css to the HTML-Newsletter
	 *
	 * @access	private
	 * @param 	text HTML newsletter
	 * @return 	boolean
	 */
	private function _addHtmlTags (&$text)
	{
		$params 			= JComponentHelper::getParams('com_bwpostman');

		$newtext = '<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.0 Transitional//EN" "http://www.w3.org/TR/xhtml1/DTD/xhtml1-transitional.dtd">
					<html xmlns="http://www.w3.org/1999/xhtml">
					<head>
						<title>HTML Newsletter</title>
  						<meta http-equiv="content-type" content="text/html; charset=utf-8" />';
		if ($params->get('use_css_for_html_newsletter')) {
			$newtext .= '<style type="text/css">'
			.$params->get('css_for_html_newsletter')
			.'</style>';
		}
		$newtext .= '</head>
  					<body>'
  					.$text
  					.'</body>
  					</html>';

 		$text = $newtext;
  		
  		return true;
	}

	/**
	 * Method to add the HTML-footer to the HTML-Newsletter
	 *
	 * @access	private
	 * @param 	text HTML newsletter
	 * @return 	boolean
	 */
	private function _addHTMLFooter(&$text) {

		$uri  				= JFactory::getURI();
		$itemid_unsubscribe	= $this->getItemid('register');
		$itemid_edit		= $this->getItemid('edit');
		$params 			= JComponentHelper::getParams('com_bwpostman');
		$impressum			= "<br /><br />" . $params->get('legal_information_text');
		$impressum			= nl2br($impressum, true);
		
		$replace = JText::_('COM_BWPOSTMAN_NL_FOOTER_HTML_LINE') . JText::sprintf('COM_BWPOSTMAN_NL_FOOTER_HTML', $uri->root(), $uri->root(), $itemid_unsubscribe, $uri->root(), $itemid_edit) . $impressum;
		
		$text = str_replace("</body>", "<div class=\"footer-outer\"><p class=\"footer-inner\">{$replace}</p></div></body>", $text);
		return true;
	}

	/**
	 * Method to add the footer Text-Newsletter
	 *
	 * @access	private
	 * @param 	text Text newsletter
	 * @return 	boolean
	 */
	private function _addTextFooter (&$text)
	{
		$uri  				= JFactory::getURI();
		$itemid_unsubscribe	= $this->getItemid('register');
		$itemid_edit		= $this->getItemid('edit');
		$params 			= JComponentHelper::getParams('com_bwpostman');
		$impressum			= "\n\n" . $params->get('legal_information_text');
		
		$replace = JText::_('COM_BWPOSTMAN_NL_FOOTER_TEXT_LINE') . JText::sprintf('COM_BWPOSTMAN_NL_FOOTER_TEXT', $uri->root(), $uri->root(), $itemid_unsubscribe, $uri->root(), $itemid_edit) . $impressum;
				
		$text = $text.$replace;
		
		return true;
	}

	/**
	 * Method to get ID of actual content ID of a newsletter from content table
	 *
	 * @access	private
	 * @param 	int 	newsletter ID
	 * @return 	int		content ID
	 */
	private function _getSingleContentId($nl_id)
	{
		$app	= JFactory::getApplication();
		$ret	= array();
		$_db	= $this->_db;
		$query	= $_db->getQuery(true);

		$query->select($_db->quoteName('id'));
		$query->from($_db->quoteName('#__bwpostman_sendmailcontent'));
		$query->where($_db->quoteName('nl_id') . ' = ' . (int) $nl_id);
		$_db->setQuery($query);

		if (!$_db->query()) {
			$app->enqueueMessage($_db->getErrorMsg(), 'error');
		}
		
		$result = $_db->loadResult(); 
		
		return $result;
	}
	
	
	/**
	 * Wenn ein Newsletter versendet werden soll, dann wird er als eine Art
	 * Archiv- und Verlaufsfunktion komplett mit Inhalt, Subject & Co. in
	 * die Tabelle sendMailContent eingefuegt
	 *
	 * @access	private
	 * @param 	int		Newsletter ID
	 * @param 	string	Recipient --> either recipients or test-recipients
	 *
	 * @return 	int		content ID, if everything went fine, else false
	 */
	private function _addSendMailContent($nl_id, $recipients)
	{
		$_db	= $this->_db;
		$query	= $_db->getQuery(true);
		
		// Get the SendmailContent ID
		$content_id = $this->_getSingleContentId($nl_id);
		
		// We load our data from newsletters table. This data is already checked for errors
		$tblNewsletters = $this->getTable('newsletters', 'BwPostmanTable');

		if ($nl_id){
			$query->select('*');
			$query->from($_db->quoteName('#__bwpostman_newsletters'));
			$query->where($_db->quoteName('id') . ' = ' . (int) $nl_id);
			
			$_db->setQuery($query);
			
			$newsletters_data = $_db->loadObject();
		}
		else {
			return false;
		}

		// Initialize the sendmailContent
		$tblSendmailContent = $this->getTable('sendmailcontent', 'BwPostmanTable');
		if ($content_id > 0) {
			$id = $content_id;
		}
		
		// Copy the data from newsletters to sendmailContent
		$tblSendmailContent->nl_id 			= $newsletters_data->id;
		$tblSendmailContent->from_name 		= $newsletters_data->from_name;
		$tblSendmailContent->from_email 	= $newsletters_data->from_email;
		$tblSendmailContent->subject 		= $newsletters_data->subject;
		$tblSendmailContent->attachment		= $newsletters_data->attachment;
		$tblSendmailContent->cc_email 		= null;
		$tblSendmailContent->bcc_email 		= null;
		$tblSendmailContent->reply_email 	= $newsletters_data->reply_email;
		$tblSendmailContent->reply_name	 	= $newsletters_data->from_name;

		// Preprocess html and text version of the newsletter
		if (!$this->_replaceLinks($newsletters_data->html_version)) return false;
		if (!$this->_replaceLinks($newsletters_data->text_version)) return false;
		if (!$this->_addHtmlTags($newsletters_data->html_version)) return false;
		 
		// We have to create two entries in the sendmailContent table. One entry for the textmail body and one for the htmlmail.
		for ($mode = 0;$mode <= 1; $mode++){

			// Set the body and the id, if exists
			if ($mode == 0) {
				$tblSendmailContent->body = $newsletters_data->text_version;
			}
			else {
				$tblSendmailContent->body = $newsletters_data->html_version;
			}

			// Set the mode (0=text,1=html)
			$tblSendmailContent->mode = $mode;

			// Store the data into the sendmailcontent-table
			// First run generates a new id, which will be used also for the second run.
			if (!$tblSendmailContent->store()) {
				return false;
			}
		}

		$id = $tblSendmailContent->id;

		return $id;
	}

	/**
	 * Method to push the recipients into a queue
	 *
	 * @access	private
	 *
	 * @param 	int		Content ID -->  --> from the sendmailcontent-Table
	 * @param 	string	Recipient --> either subscribers or test-recipients
	 * @param 	int		Newsletter ID
	 * @param	boolean	Status --> 0 = do not send to unconfirmed, 1 = sent also to unconfirmed
	 *
	 * @return 	boolean False if there occured an error
	 */
	private function _addSendMailQueue($content_id, $recipients, $nl_id, $send_to_unconfirmed)
	{
		$_db	= $this->_db;
		$query	= $_db->getQuery(true);
		
		if (!$content_id) return false;

		if (!$nl_id){
			echo JText::_('COM_BWPOSTMAN_NL_ERROR_SENDING_TECHNICAL_REASON');
			return false;
		}

		switch ($recipients){
			case "recipients": // Contain subscribers and joomla users
				$tblSendmailQueue = $this->getTable('sendmailqueue', 'BwPostmanTable');
				
				$query->select($_db->quoteName('mailinglist_id'));
				$query->from($_db->quoteName('#__bwpostman_newsletters_mailinglists'));
				$query->where($_db->quoteName('newsletter_id') . ' = ' . (int) $nl_id);
				
				$_db->setQuery($query);
				
				$mailinglists = $_db->loadObjectList();

				if (!$mailinglists) {
					return false;
				}

				$send_subscribers = 0;
				$send_to_all = 0;
				$users = array();

				foreach ($mailinglists as $mailinglist){
					$mailinglist_id = $mailinglist->mailinglist_id;
					// Mailinglists
					if ($mailinglist_id > 0) $send_subscribers = 1;
					// All subscribers
					if ($mailinglist_id == -1) {
						$send_to_all = 1;
					}
					else {
						// Usergroups
						if ((int) $mailinglist_id < 0) $users[] = -(int) $mailinglist_id;
					}
				}
				
				if ($send_to_all) {
					if ($send_to_unconfirmed) {
						$status = '0,1,9';
					}
					else {
						$status = '1,9';
					}
					if (!$tblSendmailQueue->pushAllSubscribers($content_id, $status)){
						echo JText::_('COM_BWPOSTMAN_NL_ERROR_SENDING_TECHNICAL_REASON');
						return false;
					}
				}

				if (count($users)){
					$params = JComponentHelper::getParams('com_bwpostman');
					if (!$tblSendmailQueue->pushJoomlaUser($content_id, $users, $params->get('default_emailformat'))){
						echo JText::_('COM_BWPOSTMAN_NL_ERROR_SENDING_TECHNICAL_REASON');
						return false;
					}
				}

				if ($send_subscribers){
					if ($send_to_unconfirmed) {
						$status = '0,1';
					}
					else {
						$status = '1';
					}
					if (!$tblSendmailQueue->pushAllFromNlId($nl_id, $content_id, $status)){
						echo JText::_('COM_BWPOSTMAN_NL_ERROR_SENDING_TECHNICAL_REASON');
						return false;
					}
				}
				break;

			case "testrecipients":
				$tblSubscribers		= $this->getTable('subscribers', 'BwPostmanTable');
				$testrecipients		= $tblSubscribers->loadTestrecipients();
				$tblSendmailQueue	= $this->getTable('sendmailqueue', 'BwPostmanTable');
				
				if(sizeof($testrecipients) > 0){
					foreach($testrecipients AS $testrecipient)
					$tblSendmailQueue->push($content_id, $testrecipient->emailformat, $testrecipient->email, $testrecipient->name, $testrecipient->firstname, $testrecipient->id);
				}
				break;

			default:
				echo JText::_('COM_BWPOSTMAN_NL_ERROR_SENDING_TECHNICAL_REASON');
		}
	}

	/**
	 * Check number of trials
	 * 
	 * param	int		trial
	 * 
	 * @return	bool	true if there are entries with number trials less than 2, otherwise false
	 * 
	 * since 1.0.3
	 */
	public function checkTrials($trial = 2)
	{
		$_db	= JFactory::getDbo();
		$query	= $_db->getQuery(true);
		
		$query->select('COUNT(' . $_db->quoteName('id') . ')');
		$query->from($_db->quoteName('#__bwpostman_sendmailqueue'));
		$query->where($_db->quoteName('trial') . ' < ' . (int) $trial);
		
		$_db->setQuery($query);

		if ($_db->loadResult() == 0) {
			return false;
		}
		else {
			return true;
		}
	}

	/**
	 * Make partial send. Send only, say like 50 newsletters and the next 50 in a next call.
	 * 
	 * @param $mailsToSend
	 * @param int 	mode --> 0 = regular sending, 1 = auto sending
	 * 
	 * @return unknown_type
	 */
	public function _sendMailsFromQueue($mailsPerStep = 100, $mode = 0)
	{
		$sendMailCounter = 0;
		echo JText::_('COM_BWPOSTMAN_NL_SENDING_PROCESS');
		if ($mode == 0) JText::_('COM_BWPOSTMAN_NL_SENDING_PROCESS');
		ob_flush();
		flush();

		while(1){
			$ret = $this->_sendMail($mode);
			if ($ret == 0){                              // Queue is empty!
				return 0;
				break;
			}
			$sendMailCounter++;
			if ($sendMailCounter >= $mailsPerStep) {     // Maximum is reached.
				return 1;
				break;
			}
		}
	}

	/**
	 * Funktion zum Senden eines Newsletters an einen Empfaenger aus der sendMailQueue.
	 * ACHTUNG! Es wird immer mit dem ersten Eintrag angefangen zu senden.
	 *
	 * @param int 	mode --> 0 = regular sending, 1 = auto sending
	 * 
	 * @return int (-1, if there was an error; 0, if no mail adresses left in the queue; 1, if one Mail was send).
	 */
	protected function _sendMail($mode = 0)
	{
		$app	= JFactory::getApplication();
		$res	= false;
		$_db	= $this->_db;
		$query	= $_db->getQuery(true);
				
		$tblSendMailQueue = $this->getTable('sendmailqueue', 'BwPostmanTable');
		$recipients_data = new stdClass();
		
		// getting object for queue and content
		if ($mode == 0) {
			$tblSendMailQueue	= $this->getTable('sendmailqueue', 'BwPostmanTable');
			$tblSendMailContent	= $this->getTable('sendmailcontent', 'BwPostmanTable');
		}
		else {
			$tblSendMailQueue	= $this->getTable('am_sendmailqueue', 'BwPostmanTable');
			$tblSendMailContent	= $this->getTable('am_sendmailcontent', 'BwPostmanTable');
		}
		
		// Get first entry from sendmailqueue
		// Nothing has been returned, so the queue should be empty
		if (!$tblSendMailQueue->pop()) return 0;
		
		$app->setUserState('com_bwpostman.newsletter.send.mode', $tblSendMailQueue->mode);
		
 		// Get Data from sendmailcontent (TODO, store data in this class to prevent from loding every time a mail will be sent)
		if ($mode == 0) { // regular sending
			$tblSendMailContent->load($tblSendMailQueue->content_id, $tblSendMailQueue->mode);
		}
		else { // auto sending
			$tblSendMailContent->load($tblSendMailQueue->am_content_id, $tblSendMailQueue->mode);
			$tblSendMailContent->content_id = $tblSendMailContent->id;
			$tblSendMailQueue->recipient = $tblSendMailQueue->email;
		}
		$tblSendMailContent->attachment = JPATH_SITE . '/' . $tblSendMailContent->attachment;

		// check if subscriber is archived
		if ($tblSendMailQueue->subscriber_id) {
			$query->from('#__bwpostman_subscribers');
			$query->select('id');
			$query->select('editlink');
			$query->select('archive_flag');
			$query->select('status');
			$query->where('id = ' . (int) $tblSendMailQueue->subscriber_id);
			$_db->setQuery($query);
			
			if (!$_db->query()) {
				$app->enqueueMessage($_db->getErrorMsg());
				return FALSE;
			}
			$recipients_data = $_db->loadObject();
			
			// if subscriber is archived delete entry from queue
			if ($recipients_data->archive_flag) {
				$query->clear();
				if ($mode == 0) {
					$query->from('#__bwpostman_sendmailqueue');
				}
				else {
					$query->from('#__bwpostman_am_sendmailqueue');
				}
				$query->delete();
				$query->where($_db->quoteName('subscriber_id') .' = ' . (int) $recipients_data->id);
				$_db->setQuery((string) $query);
				
				$_db->query();
									
				return 1;
			}
		} // end archived-check

		$body = $tblSendMailContent->body;
		// Replace the links to provide the correct preview
		if ($tblSendMailQueue->mode == 1) { // HTML newsletter
			if ($tblSendMailQueue->subscriber_id) { // Add footer only if it is a subscriber
				$this->_addHTMLFooter($body);
			}
		} else { // Text newsletter
			if ($tblSendMailQueue->subscriber_id) {	// Add footer only if it is a subscriber
				$this->_addTextFooter($body);
			}
		}
		$this->_replaceLinks($body);
		
		$fullname = '';
		if ($tblSendMailQueue->firstname != '') $fullname = $tblSendMailQueue->firstname . ' ';
		if ($tblSendMailQueue->name != '') $fullname .= $tblSendMailQueue->name;
		$fullname = trim($fullname);
		
		// Replace the dummies
		$body = str_replace("[NAME]", $tblSendMailQueue->name, $body);
		$body = str_replace("[LASTNAME]", $tblSendMailQueue->name, $body);
		$body = str_replace("[FIRSTNAME]", $tblSendMailQueue->firstname, $body);
		$body = str_replace("[FULLNAME]", $fullname, $body);
		$body = str_replace("[UNSUBSCRIBE_EMAIL]", $tblSendMailQueue->recipient, $body);
		
		if (property_exists($recipients_data, 'editlink')) {
			$body = str_replace("[UNSUBSCRIBE_CODE]", $recipients_data->editlink, $body);
			$body = str_replace("[EDITLINK]", $recipients_data->editlink, $body);
		}
				
		// SendMail
		if ($mode == 0) echo "\n<br>{$tblSendMailQueue->recipient} (".JText::_('COM_BWPOSTMAN_NL_ERROR_SENDING_TRIAL').($tblSendMailQueue->trial + 1).") ... ";
		ob_flush();
		flush();

		// Get a JMail instance
		$mailer		= JFactory::getMailer();
		$sender		= array();
		$reply		= array();
		
		$sender[0]	= $tblSendMailContent->from_email;
		$sender[1]	= $tblSendMailContent->from_name;

		$reply[0]	= $tblSendMailContent->reply_email;
		$reply[1]	= $tblSendMailContent->reply_name;
				
		$mailer->setSender($sender);
		$mailer->addReplyTo($reply);
		$mailer->addRecipient($tblSendMailQueue->recipient);
		$mailer->setSubject($tblSendMailContent->subject);
		$mailer->setBody($body);
		$mailer->addAttachment($tblSendMailContent->attachment);

		if ($tblSendMailQueue->mode == 1) {
			$mailer->isHTML(true);
			$mailer->Encoding = 'base64';
		}
		
		$res = $mailer->Send();
//		$res	= false;
				
		if ($res === true) {
			if ($mode == 0) {
				echo JText::_('COM_BWPOSTMAN_NL_SENT_SUCCESSFULLY');
			}
			else {
				// Sendmail was successfull, flag "sent" in table AmContent has to be set
				$tblSendMailContent->setSent($tblSendMailContent->id);
				// and test-entries may be deleted
				if (($mode == 1) && ($recipients_data->status == 9)) {
				}
			}
		}
		else{
			// Sendmail was not successfull, we need to add the recipient to the queue again.
			echo JText::_('COM_BWPOSTMAN_NL_ERROR_SENDING');
					if ($mode == 0) {
					$tblSendMailQueue->push($tblSendMailQueue->content_id, $tblSendMailQueue->mode, $tblSendMailQueue->recipient, $tblSendMailQueue->name, $tblSendMailQueue->firstname, $tblSendMailQueue->subscriber_id, $tblSendMailQueue->trial + 1);
				}
				else {
					$tblSendMailQueue->push($tblSendMailQueue->am_content_id, $tblSendMailQueue->mode, $tblSendMailQueue->email, $tblSendMailQueue->name, $tblSendMailQueue->firstname, $tblSendMailQueue->subscriber_id, $tblSendMailQueue->trial + 1);
				}
			return -1;
		}
		return 1;
	}
}

/**
 * Content Renderer Class
 * Provides methodes render the selected contents from which the newsletters shall be generated
 * --> Refering to BwPostman 1.6 beta and Communicator 2.0.0rc1
 *
 * @package		BwPostman-Admin
 * @subpackage	Newsletters
 */
class contentRenderer
{
	/**
	 * Method to get the menu item ID for the content item
	 *
	 * @access	public
	 * @return 	int menu item ID
	 */
	public function getItemid($row)
	{
		$_db	= JFactory::getDbo();
		$query	= $_db->getQuery(true);
		
		$query->select($_db->quoteName('id'));
		$query->from($_db->quoteName('#__menu'));
		$query->where($_db->quoteName('link') . ' = ' . $_db->Quote('index.php?option=com_bwpostman&view=' . $row));
		$query->where($_db->quoteName('published') . ' = ' . (int) 1);
		
		$_db->setQuery($query);
		
		$itemid = $_db->loadResult();

		if (empty($itemid)) {
			$query	= $_db->getQuery(true);
			
			$query->select($_db->quoteName('id'));
			$query->from($_db->quoteName('#__menu'));
			$query->where($_db->quoteName('link') . ' = ' . $_db->Quote('index.php?option=com_bwpostman&view=register'));
			$query->where($_db->quoteName('published') . ' = ' . (int) 1);
			
			$_db->setQuery($query);

			$itemid = $_db->loadResult();
		}

		return $itemid;
	}

	/**
	 * This is the main function to render the content from an ID to HTML
	 *
	 * @param array		$nl_content
	 * @param string	$nl_subject
	 * @return string	content
	 */
	public function getContent($nl_content, $nl_subject) {
		global $params;
		
		$param = JComponentHelper::getParams('com_bwpostman');

		$content['html_version'] = '<div class="outer"><div class="header"><img class="logo" src="'.JRoute::_(JURI::root().$param->get('logo')).'" alt="" /></div><div class="content-outer"><div class="content"><div class="content-inner"><p class="nl-intro">&nbsp;</p>';
		
		$content['text_version'] = '';
		
		if ($nl_content == null) {
			$content['html_version'] .= '';
			$content['text_version'] .= '';
				
		}
		else {
			foreach($nl_content as $content_id){
				$content['html_version'] .= $this->replaceContentHtml($content_id);
				$content['text_version'] .= $this->replaceContentText($content_id);
			}
		}
		
		$content['html_version'] .= '</div></div></div></div>';
		return $content;

	}

	public function retrieveContent($id) {
		$app	= JFactory::getApplication();
		$_db	= JFactory::getDbo();
		$query	= $_db->getQuery(true);
		
		$query->select($_db->quoteName('a') . '.*');
		$query->select('ROUND(v.rating_sum/v.rating_count) AS ' . $_db->quoteName('rating'));
		$query->select($_db->quoteName('v') . '.' . $_db->quoteName('rating_count'));
		$query->select($_db->quoteName('u') . '.' . $_db->quoteName('name') . ' AS ' . $_db->quoteName('author'));
		$query->select($_db->quoteName('cc') . '.' . $_db->quoteName('title') . ' AS ' . $_db->quoteName('category'));
		$query->select($_db->quoteName('s') . '.' . $_db->quoteName('title') . ' AS ' . $_db->quoteName('section'));
		$query->select($_db->quoteName('g') . '.' . $_db->quoteName('title') . ' AS ' . $_db->quoteName('groups'));
		$query->select($_db->quoteName('s') . '.' . $_db->quoteName('published') . ' AS ' . $_db->quoteName('sec_pub'));
		$query->select($_db->quoteName('cc') . '.' . $_db->quoteName('published') . ' AS ' . $_db->quoteName('cat_pub'));
		$query->from($_db->quoteName('#__content') . ' AS ' . $_db->quoteName('a'));
		$query->join('LEFT', $_db->quoteName('#__categories') . ' AS ' . $_db->quoteName('cc') . ' ON ' . $_db->quoteName('cc') . '.' . $_db->quoteName('id') . ' = ' . $_db->quoteName('a') . '.' . $_db->quoteName('catid'));
		$query->join('LEFT', $_db->quoteName('#__categories') . ' AS ' . $_db->quoteName('s') . ' ON ' . $_db->quoteName('s') . '.' . $_db->quoteName('id') . ' = ' . $_db->quoteName('cc') . '.' . $_db->quoteName('parent_id') . 'AND' . $_db->quoteName('s') . '.' . $_db->quoteName('extension') . ' = ' . $_db->Quote('com_content'));
		$query->join('LEFT', $_db->quoteName('#__users') . ' AS ' . $_db->quoteName('u') . ' ON ' . $_db->quoteName('u') . '.' . $_db->quoteName('id') . ' = ' . $_db->quoteName('a') . '.' . $_db->quoteName('created_by'));
		$query->join('LEFT', $_db->quoteName('#__content_rating') . ' AS ' . $_db->quoteName('v') . ' ON ' . $_db->quoteName('a') . '.' . $_db->quoteName('id') . ' = ' . $_db->quoteName('v') . '.' . $_db->quoteName('content_id'));
		$query->join('LEFT', $_db->quoteName('#__usergroups') . ' AS ' . $_db->quoteName('g') . ' ON ' . $_db->quoteName('a') . '.' . $_db->quoteName('access') . ' = ' . $_db->quoteName('g') . '.' . $_db->quoteName('id'));
		$query->where($_db->quoteName('a') . '.' . $_db->quoteName('id') . ' = ' . (int) $id);
		
		$_db->setQuery($query);
		$row = $_db->loadObject();

		if($row) {
			$params = new JRegistry();
			$params->loadString($row->attribs, 'JSON');
				
			$params->def('link_titles',	$app->getCfg('link_titles'));
			$params->def('author', 		$params->get('newsletter_show_author'));
			$params->def('createdate', 	$params->get('newsletter_show_createdate'));
			$params->def('modifydate', 	!$app->getCfg('hideModifyDate'));
			$params->def('print', 		!$app->getCfg('hidePrint'));
			$params->def('pdf', 		!$app->getCfg('hidePdf'));
			$params->def('email', 		!$app->getCfg('hideEmail'));
			$params->def('rating', 		$app->getCfg('vote'));
			$params->def('icons', 		$app->getCfg('icons'));
			$params->def('readmore', 	$app->getCfg('readmore'));
			$params->def('item_title', 	1);
				
			$params->set('intro_only', 	1);
			$params->set('item_navigation', 0);
				
			$params->def('back_button', 	0);
			$params->def('image', 			1);
				
			$row->params = $params;
			$row->text = $row->introtext;
		}
		return $row;
	}

	public function replaceContentHtml($id)
	{
		$app		= JFactory::getApplication();
		$content	= '';
			
		if($id != 0){

			// Editor user type check
			$access = new stdClass();
			$access->canEdit = $access->canEditOwn = $access->canPublish = 0;
				
			// $id = "-1" if no content is selected
			if ($id == '-1'){
				$content	.= '<h2>' . JText::_('COM_BWPOSTMAN_NL_PLACEHOLDER_TITLE') . '</h2>';
				$content	.= '<div class="intro_text">' . JText::_('COM_BWPOSTMAN_NL_PLACEHOLDER_CONTENT') . '</div>';
				return stripslashes($content);
			}

			$row = $this->retrieveContent($id);
				
			if ($row) {
				$params = $row->params;
				$model		= new BwPostmanModelNewsletter;
				$lang		= $model->getArticleLanguage($row->id);
				$_Itemid	= ContentHelperRoute::getArticleRoute($row->id, 0, $lang);
				$link		= JRoute::_(JURI::base());
				if ($_Itemid) $link .= $_Itemid;
				
				$app->triggerEvent('onPrepareContent', array(&$row, &$params, 0), true);

				$intro_text = $row->text;

				if (intval($row->created) != 0) {
					$create_date = JHTML::_('date', $row->created);
				}
				$html_content = new HTML_content();

				ob_start();
				// Displays Item Title
				$html_content->Title($row, $params, $access);

				$content .= ob_get_contents();
				ob_end_clean();
				// Displays Category
				ob_start();
				$html_content->Category($row, $params);

				// Displays Author Name
				$html_content->Author($row, $params);

				// Displays Created Date
				$html_content->CreateDate($row, $params);

				// Displays Urls
				$html_content->URL($row, $params);
				$content .= ob_get_contents();
				ob_end_clean();

				$content .= '<div class="intro_text">'
				. $intro_text //(function_exists('ampReplace') ? ampReplace($intro_text) : $intro_text). '</td>'
				. '</div>'
				. '<div class="read_on">'
				. '		<p>'
				. '		<a href="'. str_replace('administrator/', '', $link) . '" class="readon">'.
				JText::_('READ_MORE')
				. '		</a><br/><br/><br/>'
				. '		</p>'
				. '	</div>';
				
				return stripslashes($content);
			}
		}
		else {
			return 'error retrieving Content ID: '.$id.'<br/>';
		}
	}

	public function replaceContentText($id){
		$app = JFactory::getApplication();

		if($id != 0){
			$row = $this->retrieveContent($id);
				
			if ($row) {
				$params = $row->params;

				$model		= new BwPostmanModelNewsletter;
				$lang		= $model->getArticleLanguage($row->id);
				$_Itemid	= ContentHelperRoute::getArticleRoute($row->id, 0, $lang);
				$link		= JRoute::_(JURI::base());
				if ($_Itemid) $link .= $_Itemid;
								
				$app->triggerEvent('onPrepareContent', array(&$row, &$params, 0), true);

				$intro_text = $row->text;
				$intro_text = strip_tags($intro_text);

				$intro_text = $this->unHTMLSpecialCharsAll($intro_text);

				if (intval($row->created) != 0) {
					$create_date = JHTML::_('date', $row->created);
				}

				$content = "\n" . $row->title;
				if (($params->get('author')) && ($row->created_by != '')) {
					$content .= "\n\n(" .  JTEXT::_('WRITTEN_BY') . " ".($row->created_by_alias ? $row->created_by_alias : $row->created_by). " )\n";
				}
				if ($params->get('createdate')) {
					$content .= $create_date;
				}
				$content .= "\n\n" . $intro_text
				. "\n\n". JTEXT::_('READ_MORE')
				. ": \n". str_replace('administrator/', '', $link) . "\n\n";

				return stripslashes($content);
			}
		}
	}

	private function unHTMLSpecialCharsAll($text) {

		$text = $this->deHTMLEntities($text);

		return $text;
	}

	/**
	 * convert html special entities to literal characters
	 */
	private function deHTMLEntities($text) {
		$search = array(
		"'&(quot|#34);'i",
		"'&(amp|#38);'i",
		"'&(lt|#60);'i",
		"'&(gt|#62);'i",
		"'&(nbsp|#160);'i",   "'&(iexcl|#161);'i",  "'&(cent|#162);'i",   "'&(pound|#163);'i",  "'&(curren|#164);'i",
		"'&(yen|#165);'i",    "'&(brvbar|#166);'i", "'&(sect|#167);'i",   "'&(uml|#168);'i",    "'&(copy|#169);'i",
		"'&(ordf|#170);'i",   "'&(laquo|#171);'i",  "'&(not|#172);'i",    "'&(shy|#173);'i",    "'&(reg|#174);'i",
		"'&(macr|#175);'i",   "'&(neg|#176);'i",    "'&(plusmn|#177);'i", "'&(sup2|#178);'i",   "'&(sup3|#179);'i",
		"'&(acute|#180);'i",  "'&(micro|#181);'i",  "'&(para|#182);'i",   "'&(middot|#183);'i", "'&(cedil|#184);'i",
		"'&(supl|#185);'i",   "'&(ordm|#186);'i",   "'&(raquo|#187);'i",  "'&(frac14|#188);'i", "'&(frac12|#189);'i",
		"'&(frac34|#190);'i", "'&(iquest|#191);'i", "'&(Agrave|#192);'",  "'&(Aacute|#193);'",  "'&(Acirc|#194);'",
		"'&(Atilde|#195);'",  "'&(Auml|#196);'",    "'&(Aring|#197);'",   "'&(AElig|#198);'",   "'&(Ccedil|#199);'",
		"'&(Egrave|#200);'",  "'&(Eacute|#201);'",  "'&(Ecirc|#202);'",   "'&(Euml|#203);'",    "'&(Igrave|#204);'",
		"'&(Iacute|#205);'",  "'&(Icirc|#206);'",   "'&(Iuml|#207);'",    "'&(ETH|#208);'",     "'&(Ntilde|#209);'",
		"'&(Ograve|#210);'",  "'&(Oacute|#211);'",  "'&(Ocirc|#212);'",   "'&(Otilde|#213);'",  "'&(Ouml|#214);'",
		"'&(times|#215);'i",  "'&(Oslash|#216);'",  "'&(Ugrave|#217);'",  "'&(Uacute|#218);'",  "'&(Ucirc|#219);'",
		"'&(Uuml|#220);'",    "'&(Yacute|#221);'",  "'&(THORN|#222);'",   "'&(szlig|#223);'",   "'&(agrave|#224);'",
		"'&(aacute|#225);'",  "'&(acirc|#226);'",   "'&(atilde|#227);'",  "'&(auml|#228);'",    "'&(aring|#229);'",
		"'&(aelig|#230);'",   "'&(ccedil|#231);'",  "'&(egrave|#232);'",  "'&(eacute|#233);'",  "'&(ecirc|#234);'",
		"'&(euml|#235);'",    "'&(igrave|#236);'",  "'&(iacute|#237);'",  "'&(icirc|#238);'",   "'&(iuml|#239);'",
		"'&(eth|#240);'",     "'&(ntilde|#241);'",  "'&(ograve|#242);'",  "'&(oacute|#243);'",  "'&(ocirc|#244);'",
		"'&(otilde|#245);'",  "'&(ouml|#246);'",    "'&(divide|#247);'i", "'&(oslash|#248);'",  "'&(ugrave|#249);'",
		"'&(uacute|#250);'",  "'&(ucirc|#251);'",   "'&(uuml|#252);'",    "'&(yacute|#253);'",  "'&(thorn|#254);'",
		"'&(yuml|#255);'");
		$replace = array(
		"\"",
		"&",
		"<",
		">",
		" ",      chr(161), chr(162), chr(163), chr(164), chr(165), chr(166), chr(167), chr(168), chr(169),
		chr(170), chr(171), chr(172), chr(173), chr(174), chr(175), chr(176), chr(177), chr(178), chr(179),
		chr(180), chr(181), chr(182), chr(183), chr(184), chr(185), chr(186), chr(187), chr(188), chr(189),
		chr(190), chr(191), chr(192), chr(193), chr(194), chr(195), chr(196), chr(197), chr(198), chr(199),
		chr(200), chr(201), chr(202), chr(203), chr(204), chr(205), chr(206), chr(207), chr(208), chr(209),
		chr(210), chr(211), chr(212), chr(213), chr(214), chr(215), chr(216), chr(217), chr(218), chr(219),
		chr(220), chr(221), chr(222), chr(223), chr(224), chr(225), chr(226), chr(227), chr(228), chr(229),
		chr(230), chr(231), chr(232), chr(233), chr(234), chr(235), chr(236), chr(237), chr(238), chr(239),
		chr(240), chr(241), chr(242), chr(243), chr(244), chr(245), chr(246), chr(247), chr(248), chr(249),
		chr(250), chr(251), chr(252), chr(253), chr(254), chr(255));
		return $text = preg_replace($search, $replace, $text);
	}

}

/**
 * Utility class for writing the HTML for content
 * --> Refering to Communicator 2.0.0rc1
 *
 * @package 		BwPostman-Admin
 * @subpackage 	Newsletters
 */
class HTML_content {
	/**
	 * Writes Title
	 */
	public function Title(&$row, &$params, &$access) {
		if ($params->get('item_title')) {
			if ($params->get('link_titles') && $row->link_on != '') {
				?>
				<h2><a href="<?php echo $row->link_on;?>"
					class="contentpagetitle<?php echo $params->get('pageclass_sfx'); ?>">
								<?php echo $row->title;?></a></h2>
								<?php
							} else {
								?>
				<h2><?php echo $row->title;?></h2>
				<?php
			}
		}
	}

	/**
	 * Writes Category
	 */
	public function Category(&$row, &$params) {
		if ($params->get('category')) {
			?>
			<span class="sc_category"> <?php
			echo $row->category;
			?> </span>
			<?php
		}
	}

	/**
	 * Writes Author name
	 */
	public function Author(&$row, &$params) {
		if (($params->get('author')) && ($row->created_by != '')) {
			?>
			<p class="created_by"> <?php echo JTEXT::_('WRITTEN_BY') . ' '.($row->created_by_alias ? $row->created_by_alias : $row->created_by); ?>
			</p> &nbsp;&nbsp;
			<?php
		}
	}


	/**
	 * Writes Create Date
	 */
	public function CreateDate(&$row, &$params) {
		$create_date = null;

		if (intval($row->created) != 0) {
			$create_date = JHTML::_('date', $row->created);
		}

		if ($params->get('createdate')) {
			?>
			<p class="createdate"><?php echo $create_date; ?></p>
			<?php
		}
	}

	/**
	 * Writes URL's
	 */
	public function URL(&$row, &$params) {
		if ($params->get('url') && $row->urls) {
			?>
			<p class="row_url"><a
				href="http://<?php echo $row->urls ; ?>" target="_blank"> <?php echo $row->urls; ?></a>
			</p>
			<?php
		}
	}

	/**
	 * Writes Modified Date
	 */
	public function ModifiedDate(&$row, &$params) {
		$mod_date = null;

		if (intval($row->modified) != 0) {
			$mod_date = JHTML::_('date', $row->modified);
		}

		if (($mod_date != '') && $params->get('modifydate')) {
			?>
			<p class="modifydate"><?php echo JTEXT::_('LAST_UPDATED'); ?>
			(<?php echo $mod_date; ?>)</p>
			<?php
		}
	}

	/**
	 * Writes Readmore Button
	 */
	public function ReadMore (&$row, &$params) {
		if ($params->get('readmore')) {
			if ($params->get('intro_only') && $row->link_text) {
				?>
				<p class="link_on"><a href="<?php echo $row->link_on;?>"
					class="readon<?php echo $params->get('pageclass_sfx'); ?>"> <?php echo $row->link_text;?></a>
				</p>
				<?php
			}
		}
	}
}